##########################################
# Pygames - Group Kollisionen 
# Autor: M. Schmid
##########################################

##########################################
# Externe Bibliotheken einbinden
##########################################
import pygame as py
from random import randint


##########################################
# Pygames initialisieren
##########################################
py.init()

# Fenstergrösse setzen
win_size = (800,800)
screen = py.display.set_mode(win_size)

# Titel des Fensters
py.display.set_caption("Pygames: Der Mausinator")

# Schrift für den Text erstellen
my_font = py.font.SysFont('Comic Sans MS', 36)
my_font_greater = py.font.SysFont('Comic Sans MS', 54)

# Sounds laden
schusse = py.mixer.Sound("sound/gun-shot.wav")
schusse.set_volume(0.3)
explosion = py.mixer.Sound("sound/explosion-012.mp3")
explosion.set_volume(0.4)

#########################################################################
# Beginn der Klassendefinitionen
#########################################################################

# Bauplan für die Maus
class Mouse(py.sprite.Sprite):
    def __init__(self): 
        super().__init__()

        self.image  = py.image.load("./res/mouse.png").convert_alpha()
        self.image = py.transform.rotate(self.image, 180)

        # Rechteck ums Bild bestimmen und Koordinaten setzen
        self.rect   = self.image.get_rect()
        self.rect.x = win_size[0] / 2
        self.rect.y = win_size[1] - self.rect.height

        # Schrittweite
        self.step_size  = 8

        # Mask definieren, für die Pixel-Kollision
        self.mask = py.mask.from_surface(self.image)


# Bauplan für die Muenze
class Coin(py.sprite.Sprite):
    def __init__(self): 
        super().__init__()

        # Bilder laden und Startbild setzen 
        self.animation = load_images("res/coin/", "coin", ".gif", 8, 50, 50)
        self.costume = randint(0, 7)
        self.image  = self.animation[self.costume]

        # Rechteck ums Bild bestimmen und Koordinaten setzen
        self.rect   = self.image.get_rect()
        self.rect.x = randint(0, win_size[0] - self.rect.width) 
        self.rect.y = randint(0, win_size[1] - self.rect.height) 

        # Mask definieren, für die Pixel-Kollision
        self.mask = py.mask.from_surface(self.image)
    
    # Das Objekt soll nach jedem Schleifendurchgang das Bild/Kostüm
    # wechseln. Dies geschieht in der "update"-Methode der Klasse
    def update(self):
        self.costume = (self.costume + 1 ) % len(self.animation)
        self.image  = self.animation[self.costume]
        self.mask = py.mask.from_surface(self.image)


# Bauplan für die Expolosion definieren
class Explosion(py.sprite.Sprite):
    def __init__(self, center):
        super().__init__()

        # Bilder laden und Startbild setzen 
        self.animation = load_images("res/explosion/", "e", ".gif", 40, 128, 128)
        self.costume = 0
        self.image = self.animation[self.costume]
        self.rect = self.image.get_rect()
        self.rect.center = center
        

    # Das Objekt soll nach jedem Schleifendurchgang das Bild/Kostüm
    # wechseln. Dies geschieht in der "update"-Methode der Klasse
    def update(self):
        self.costume = (self.costume + 1) % 40
        self.image = self.animation[self.costume]
        if self.costume > 38:
            self.kill()

# Bauplan für die Schüsse der Maus definieren
class Bullet(py.sprite.Sprite):
    def __init__(self, x, y):
        super().__init__()
        self.image = py.image.load("res/bullet.png").convert()
        self.rect = self.image.get_rect()
        self.rect.centerx = x
        self.rect.y = y
        self.step_size = 20
              
    # Das Objekt bewegt sich mit einer konstanten Geschwindigkeit,
    # d.h., dass die Koordinaten aktualsiert werden müssen
    def update(self):
        # Schuss nach "oben" bewegen
        self.rect.y = self.rect.y - self.step_size
        # Ist der Schuss oben angekommen?
        # Den Schuss aus dem Speicher löschen
        if self.rect.y + self.rect.height < 0:                           
            self.kill()                                                  


###########################################
# Globale Funktionen definieren
##########################################

# Funktion, die die Tasten-Events überprüft
def key_handler():
    key = py.key.get_pressed()
    
    # Bedienung Spieler 1
    if key[py.K_LEFT]:
        if maus.rect.x - maus.step_size <= 0:
            maus.rect.x = 0
        else:
            maus.rect.x -= maus.step_size
    if key[py.K_RIGHT]:
        if maus.rect.x + maus.step_size + maus.rect.width >= win_size[0]:
            maus.rect.x = win_size[0] - maus.rect.width
        else:
            maus.rect.x += maus.step_size


# Funktion: "load_images",
# die alle Bilder für eine Klasse lädt
#
# Eingaben:
#   - path = Pfad zum Ordner, wo die Bilder drin sind
#   - names = Bildernamen ohne Nummerierung
#   - ending = Bildformat, z.B. ".gif" oder anderes
#   - number = Anzahl Bilder, die geladen werden sollen
#   - xpix,ypix = Groesse des Bildes im Spiel 
#
# Ausgabe:
#   - Liste mit Pygame-Bildern
def load_images(path,names,ending,number,xpix,ypix):
    # In Animation werden die Pygame-Bilder gespeichert
    animation = []

    for i in range(number):
        file_name = path + names + str(i) + ending
        img = py.image.load(file_name).convert_alpha() 
        animation.append(py.transform.scale(img, (xpix, ypix)))

    return animation

# Funktion: "draw_score",
# die den aktuellen Kontostand im Fenster ausgibt
def draw_score(anzahl, zeit):
    text_string = "Anzahl verbleibende Münzen: " + str(anzahl)
    text_string += "   Zeit: " + str(zeit)
    text = my_font.render(text_string, True, [0, 0, 0]) 
    
    screen.blit(text, (win_size[0] / 2 - text.get_rect().width / 2 ,20))


def draw_ende(zeit):
    text_string = "Ende! Zeit: " + str(zeit) + " Sekunden"
    text = my_font_greater.render(text_string, True, [255, 0, 0]) 
    screen.blit(text, (win_size[0] * 0.2 , win_size[1] / 2 - text.get_rect().height))
    
    # Press ESC
    text = my_font.render("Close Window or press 'ESC' to end the game.", True, [0, 0, 0])
    screen.blit(text, (win_size[0] * 0.2 , win_size[1] / 2 + text.get_rect().height))

# Funktion: "check_collisions",
# die überprüft, ob die Bullets eine Muenze beruehren
# Bei einer Kollision, wird die Muenze und das Bullet entfernt
def is_a_collision():
    kollisionsliste = py.sprite.groupcollide(all_bullets, all_munzen, True, True, collided = py.sprite.collide_mask)
    if len(kollisionsliste) != 0:
        for i in kollisionsliste.values():
            for j in i:
                all_sprites.add(Explosion(j.rect.center))
        return True
    else:
        return False


##########################################
# Globale Variabeln initialisieren
##########################################
game_is_running = True

# Eine Pygame-Uhr um die Framerate zu kontrollieren
clock = py.time.Clock()

# Framerate: fps = frames per second
fps = 24

# Maus-Spieler wird erstellt
maus = Mouse()

# Gruppen für die Sprite Objekte erstellen
all_sprites = py.sprite.Group()
all_sprites.add(maus)

# Münzen
all_munzen = py.sprite.Group()
anzahl_munzen = 6
for i in range(anzahl_munzen):
    all_munzen.add(Coin())

all_sprites.add(all_munzen)

# Bullets
all_bullets = py.sprite.Group()

# Startzeit für den Counter auf dem Bildschirm
start = py.time.get_ticks()

# Bildschirme:
#   - 0: Bildschirm während des Spiels
#   - 1: Endschirm, erscheint wenn alle Muenzen eingesammelt wurden
which_window = 0
##########################################
# Hauptschleife
##########################################
while game_is_running:
    
    ##########################################
    # Events abfragen
    ##########################################
    for event in py.event.get():
        # Wird das Fenster geschlossen, soll
        # auch das Spiel beendet werden
        if event.type == py.QUIT or (event.type == py.KEYDOWN and event.key == py.K_ESCAPE):
            game_is_running = False
        elif event.type == py.KEYDOWN and event.key == py.K_SPACE:
            bullet = Bullet(maus.rect.centerx - 7, maus.rect.y)
            bullet.rect.y -= bullet.rect.height
            all_bullets.add(bullet)
            all_sprites.add(bullet)
            py.mixer.Sound.play(schusse)

    if which_window == 0:  
        # Zeit messen (in Sekunden)
        counter = (py.time.get_ticks() - start) // 1000

        # Game - Bildschirm
        ##########################################
        # Update Game State
        ##########################################

        # Wurde eine Taste gedrückt?
        key_handler()
        
        # Hat eine Beruehrung stattgefunden?
        if is_a_collision():
            anzahl_munzen = len(all_munzen)
            py.mixer.Sound.play(explosion)

        # Den Game-State der Sprites aktualisieren
        # Ruft auf allen Sprites die "update"-Funktion auf
        all_sprites.update()

        ##########################################
        # Alles zeichnen
        ##########################################
        # Hintergrund zeichnen = alles löschen
        screen.fill([255,255,255])

        # Alle Sprites in der jeweiligen Gruppe zeichnen
        all_sprites.draw(screen)

        # Texte darstellen
        draw_score(anzahl_munzen, counter)

        if anzahl_munzen == 0:
            which_window += 1
            time_needed = counter
    else:
        # Endbildschirm
        ##########################################
        # Alles zeichnen
        ##########################################
        screen.fill([255,255,255])
        draw_ende(time_needed)

    py.display.update()
    clock.tick(fps)
   
##########################################
# Hauptschleife beendet -> Game Over
##########################################

##########################################
# Spiel ist beendet
##########################################
py.quit()